/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.websocket;

import io.jsonwebtoken.JwtException;
import org.apache.pulsar.client.api.SubscriptionType;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.TopicName;

import java.util.*;

public class RelNotifToken {

    private static final Set<String> binaryNamespaces = new HashSet<>(Arrays.asList("mqtt"));

    private final TopicName topicName;
    private final String subscriptionName;
    private final boolean isReadOnly, isWriteOnly;
    private boolean isBinaryProtocolOnly;
    private final boolean sharedSubscription;
    private final boolean isVolatile;

    public RelNotifToken(String tokenString, String keyFile) {
        Map<String, Object> claims = JwtClaimsHelper.getJwtClaims(tokenString, keyFile);
        try {
            String[] topic = ((String) claims.get("topic")).split("/");

            subscriptionName = (String) claims.get("sub");
            isReadOnly = "true".equals(claims.get("readOnly"));
            isWriteOnly = "true".equals(claims.get("writeOnly"));
            sharedSubscription = "true".equals(claims.get("shared"));
            isVolatile = "true".equals(claims.get("volatile"));
            // boolean isWebSocketOnly = "true".equals(claims.get("webSocketOnly"));
            isBinaryProtocolOnly = "true".equals(claims.get("binaryOnly"));

            topicName = TopicName.get(
                    isVolatile? "non-persistent" : "persistent",
                    NamespaceName.get(topic[0], topic[1]),
                    topic[2]);
        } catch (Exception e) {
            throw new JwtException("Invalid claims in token");
        }
    }

    public static boolean isBinaryNamespace(String namespace) {
        return binaryNamespaces.contains(namespace);
    }

    public void verifyCanRead() {
        if (isWriteOnly) {
            throw new JwtAccessModeException("Write only token");
        }
    }

    public void verifyCanWrite() {
        if (isReadOnly) {
            throw new JwtAccessModeException("Read only token");
        }
    }

    public void verifyWebSocketAllowed() {
        if (isBinaryProtocolOnly) {
            throw new JwtAccessModeException("Binary protocol only token");
        }
    }

    public TopicName getTopicName() {
        return topicName;
    }

    public String getSubscriptionName() {
        return subscriptionName;
    }

    public Optional<SubscriptionType> getSubscriptionType() {
        if (sharedSubscription) {
            return Optional.of(SubscriptionType.Key_Shared);
        }
        return Optional.empty();
    }

    public boolean isVolatile() {
        return isVolatile;
    }

    // Performance params are to be added. Possibly the token has a class of service and the params carry requests?

    public Optional<String> getAckTimeoutMillis(Map<String, String> queryParams) {
        return Optional.empty();
    }

    public Optional<Integer> getReceiverQueueSize(Map<String, String> queryParams) {
        return Optional.empty();
    }

    public Optional<Integer> getBatchingMaxMessages(Map<String, String> queryParams) {
        return Optional.empty();
    }

    public Optional<Integer> getMaxPendingMessages(Map<String, String> queryParams) {
        return Optional.empty();
    }
}
