/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.websocket;

import org.apache.pulsar.broker.PulsarServerException;
import org.apache.pulsar.client.admin.PulsarAdmin;
import org.apache.pulsar.client.admin.PulsarAdminException;
import org.apache.pulsar.client.api.PulsarClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class RelNotifControlServlet extends HttpServlet {

    public static final String SERVLET_PATH = "/notification2/control"; // c8y path

    private static final long serialVersionUID = 1L;
    private final transient WebSocketService service;
    private static final Logger log = LoggerFactory.getLogger(RelNotifControlServlet.class);
    private static ExecutorService provisionerExecutor = Executors.newSingleThreadExecutor();

    public RelNotifControlServlet(WebSocketService service) {
        this.service = service;
    }

    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        resp.setContentType("application/json;charset=utf-8");

        String pathInfo = req.getPathInfo();

        Map<String, String[]> params = req.getParameterMap();

        if ("/unsubscribe".equals(pathInfo)) {
            unsubscribe(resp, params);
        } else {
            throw new ServletException("Unknown c8y notification2/control request");
        }
    }

    private PulsarAdmin getPulsarAdmin() throws PulsarServerException, PulsarClientException {
        String adminUrl = service.getServiceUrl();
        return PulsarAdmin.builder().serviceHttpUrl(adminUrl).build();
    }

    private void unsubscribe(final HttpServletResponse resp, final Map<String, String[]> params) throws ServletException, IOException {

        final TokenInfo token = validateToken(params);

        final boolean force = getBooleanParam(params, "force");
        final boolean background = getBooleanParam(params, "background");

        String result;
        if (background) {
            provisionerExecutor.execute(() -> {
                unsubscribe(token, force);
            });
            result = "{ \"result\": \"scheduled\" }";
        } else {
            Optional<Exception> exn = unsubscribe(token, force);
            if (exn.isPresent()) {
                resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, "Unsubscribe failed with: " + exn);
                return;
            }
            result = "{ \"result\": \"done\" }";
        }
        resp.getWriter().print(result);
    }

    private Optional<Exception> unsubscribe(TokenInfo token, boolean force) {
        try {
            PulsarAdmin pulsarAdmin = getPulsarAdmin();

            try {
                pulsarAdmin.topics().deleteSubscription(token.tenantName + "/" + token.namespaceName + "/" + token.topicName, token.subscription, force);
            } finally {
                pulsarAdmin.close();
            }
        } catch (PulsarAdminException | PulsarServerException | PulsarClientException exn) {
            log.error("Unsubscribe subscriber failed: ", exn);
            return Optional.of(exn);
        }
        return Optional.empty();
    }

    private boolean getBooleanParam(Map<String, String[]> params, String which) {
        boolean param = false;
        if (params.containsKey(which)) {
            String forceValue = params.get(which)[0];
            if ("true".equals(forceValue) || "1".equals(forceValue)) {
                param = true;
            }
        }
        return param;
    }

    private TokenInfo validateToken(Map<String, String[]> params) throws ServletException {
        if (!params.containsKey("token")) {
            throw new ServletException("Expected token query parameter");
        }
        String token = params.get("token")[0];

        String keyFile = service.getConfig().getWebSocketPublicKeyFile();

        Map<String, Object> claims;
        try {
            claims = JwtClaimsHelper.getJwtClaims(token, keyFile);
        } catch (Exception e) {
            throw new ServletException("Failed to verify token " + e.getMessage(), e);
        }

        String topic = (String) claims.get("topic");
        if (topic == null) {
            throw new ServletException("Expected topic in token");
        }
        String sub = (String) claims.get("sub");
        if (sub == null) {
            throw new ServletException("Expected sub in token");
        }

        return new TokenInfo(topic, sub);
    }

    private static class TokenInfo {
        String subscription;
        String tenantName;
        String namespaceName;
        String topicName;

        TokenInfo(String topic, String subscription) throws ServletException {
            this.subscription = subscription;

            String[] topicParts = topic.split("/");
            if (topicParts.length != 3) {
                throw new ServletException("Expected tenant/namespace/topic in token");
            }
            tenantName = topicParts[0];
            namespaceName = topicParts[1];
            topicName = topicParts[2];
        }
    }
}
