/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.websocket;

import io.jsonwebtoken.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.*;

import static java.lang.String.format;
import static org.bouncycastle.util.encoders.Hex.toHexString;

public class JwtClaimsHelper {

    private static final Logger log = LoggerFactory.getLogger(JwtClaimsHelper.class);
    private static PublicKey publicKey = null;
    private static String publicKeySignature;

    public static Map<String, Object> getJwtClaims(String jws, String keyFile) {
        try {
            return getJwtClaims(jws, getPublicKey(keyFile));
        } catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException | InvalidKeyException |
                 JwtException exn) {
            throw new JwtException(format(
                    "Exception when verifying claims of Notifications 2.0 token ending %s (public key SHA256 signature: %s): %s",
                    tokenEnd(jws), publicKeySignature, exn.getMessage()), exn);
        }
    }

    private static String tokenEnd(String token) {
        return token.length() < 10? token: token.substring(token.length() - 10);
    }

    private static Map<String, Object> getJwtClaims(String jws, PublicKey publicKey) {
        log.info("Parsing Notifications 2.0 token ending {} (public key SHA256 signature: {})", tokenEnd(jws), publicKeySignature);
        Jws<Claims> jwt = Jwts.parserBuilder()
                .setSigningKey(publicKey)
                .build()
                .parseClaimsJws(jws);
        return jwt.getBody();
    }

    private static synchronized PublicKey getPublicKey(String keyFile) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException, InvalidKeyException {
        if (publicKey == null ) {
            final byte[] keyFileBytes = Files.readAllBytes(Paths.get(keyFile));
            final String keyString = new String(keyFileBytes, StandardCharsets.US_ASCII);
            final String strippedKeyString = keyString.trim()
                    .replace("-----BEGIN PUBLIC KEY-----\n", "")
                    .replace("\n-----END PUBLIC KEY-----", "");

            final X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(Base64.getMimeDecoder().decode(strippedKeyString));
            final KeyFactory pkf2 = KeyFactory.getInstance("RSA");
            publicKey = pkf2.generatePublic(publicKeySpec);

            publicKeySignature =toHexString(MessageDigest.getInstance("SHA-256").digest(keyFileBytes));
            log.info("Notifications 2.0 public key loaded: SHA256 signature: {}", publicKeySignature);
        }
        return publicKey;
    }
}
