/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.proxy.server;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.ws.rs.core.Response;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.common.api.AuthData;
import org.apache.pulsar.common.api.proto.CommandConnect;
import org.apache.pulsar.proxy.server.ProxyService;
import org.apache.pulsar.websocket.JwtClaimsHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Claims {
    private static final Logger LOG = LoggerFactory.getLogger(Claims.class);
    private static final int HTTP_TIMEOUT_SECONDS = 10;
    private static final HttpClient C8Y_HTTP_CLIENT = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).build();
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final Map<String, Object> claims;

    Claims(Map<String, Object> claims) {
        this.claims = claims;
    }

    public static Claims authenticateC8Y(ProxyService service, CommandConnect connect) throws PulsarClientException {
        if (connect.hasAuthMethodName() && "token".equals(connect.getAuthMethodName())) {
            return Claims.authenticateC8YToken(service, connect);
        }
        if (connect.hasAuthMethodName() && "basic".equals(connect.getAuthMethodName())) {
            return Claims.authenticateC8YBasic(service, connect);
        }
        throw new PulsarClientException.UnsupportedAuthenticationException("Must include a Cumulocity token or use 'basic' authentication method");
    }

    private static Claims authenticateC8YBasic(ProxyService service, CommandConnect connect) throws PulsarClientException {
        byte[] authData = connect.getAuthData();
        if (ArrayUtils.isEmpty((byte[])authData)) {
            throw new PulsarClientException.UnsupportedAuthenticationException("Authentication failed: Cumulocity basic authentication credentials must not be empty.");
        }
        String credentials = new String(AuthData.of((byte[])authData).getBytes());
        URI baseUri = URI.create(service.getConfiguration().getCumulocityBaseUrl());
        if (baseUri.getScheme() == null) {
            baseUri = URI.create("http://" + baseUri.toString());
        }
        URI authUri = baseUri.resolve(service.getConfiguration().getCumulocityAuthenticationUrl());
        LOG.debug("Attempting C8Y basic auth. Base URL: <{}>, Auth URL: <{}>", (Object)service.getConfiguration().getCumulocityBaseUrl(), (Object)service.getConfiguration().getCumulocityAuthenticationUrl());
        try {
            HttpRequest request = Claims.buildRequest(credentials, authUri);
            HttpResponse<String> response = C8Y_HTTP_CLIENT.send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != Response.Status.OK.getStatusCode()) {
                throw new PulsarClientException.AuthenticationException("Cumulocity basic authentication failed with status code: " + response.statusCode());
            }
            AuthorizedTopics parsedResponse = (AuthorizedTopics)OBJECT_MAPPER.readValue(response.body(), AuthorizedTopics.class);
            HashMap<String, Object> claimsMap = new HashMap<String, Object>();
            claimsMap.put("isBasicAuth", true);
            claimsMap.put("topics", parsedResponse.getTopics());
            claimsMap.put("username", Claims.extractUsername(credentials));
            Claims.logAuthEvent(claimsMap);
            return new Claims(claimsMap);
        }
        catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            LOG.error("Error during Cumulocity basic authentication request", (Throwable)e);
            throw new PulsarClientException.AuthenticationException("A error occurred during basic authentication: " + e.getMessage());
        }
    }

    private static String getAuthenticationString(String credentials) {
        String encoded = Base64.getEncoder().encodeToString(credentials.getBytes(StandardCharsets.UTF_8));
        return "Basic " + encoded;
    }

    private static HttpRequest buildRequest(String credentials, URI uri) {
        return HttpRequest.newBuilder().timeout(Duration.ofSeconds(10L)).uri(uri).header("Accept", "application/json").header("Authorization", Claims.getAuthenticationString(credentials)).build();
    }

    public static Claims authenticateC8YToken(ProxyService service, CommandConnect connect) throws PulsarClientException.UnsupportedAuthenticationException {
        if (StringUtils.isEmpty((String)service.getConfiguration().getProxyPublicKeyFile())) {
            return null;
        }
        if (connect.hasAuthMethodName() && "token".equals(connect.getAuthMethodName())) {
            byte[] authData = connect.getAuthData();
            if (authData == null || authData.length == 0) {
                throw new PulsarClientException.UnsupportedAuthenticationException("Must include a non-empty Cumulocity token");
            }
            AuthData tokenAuthData = AuthData.of((byte[])authData);
            String jws = new String(tokenAuthData.getBytes());
            Map claimsMap = JwtClaimsHelper.getJwtClaims((String)jws, (String)service.getConfiguration().getProxyPublicKeyFile());
            Claims.logAuthEvent(claimsMap);
            return new Claims(claimsMap);
        }
        throw new PulsarClientException.UnsupportedAuthenticationException("Must include a Cumulocity token");
    }

    private static String extractUsername(String credentials) {
        int idx = credentials.indexOf(58);
        return idx > 0 ? credentials.substring(0, idx) : credentials;
    }

    private static void logAuthEvent(Map<String, Object> claimsMap) {
        if (Boolean.TRUE.equals(claimsMap.get("isBasicAuth"))) {
            LOG.info("C8Y basic authentication successful for user [{}]", claimsMap.get("username"));
            return;
        }
        LOG.info("Proxy connection with C8Y token for topic {} and subscriber {}", claimsMap.get("topic"), claimsMap.get("sub"));
    }

    public boolean hasAnyPermissionFor(String topic) {
        if (this.isBasicAuth()) {
            return this.checkBasicAuthPermission(topic, "read") || this.checkBasicAuthPermission(topic, "write");
        }
        return this.forTopic(topic);
    }

    public boolean canProduce(String topic) {
        if (this.isBasicAuth()) {
            return this.checkBasicAuthPermission(topic, "write");
        }
        return this.forTopic(topic) && !this.isReadOnly();
    }

    public boolean canConsume(String topic) {
        if (this.isBasicAuth()) {
            return this.checkBasicAuthPermission(topic, "read");
        }
        return this.forTopic(topic) && !this.isWriteOnly();
    }

    private boolean checkBasicAuthPermission(String requestedTopic, String requiredPermission) {
        List topics = (List)this.claims.get("topics");
        if (topics == null || topics.isEmpty()) {
            return false;
        }
        String cleanTopicName = Claims.cleanTopicName(requestedTopic);
        return topics.stream().filter(authorizedTopic -> this.matchesPartitioned(cleanTopicName, authorizedTopic.getName())).map(AuthorizedTopic::getPermissions).filter(Objects::nonNull).anyMatch(permissions -> permissions.contains(requiredPermission));
    }

    private static String cleanTopicName(String topic) {
        if (topic.startsWith("persistent://")) {
            return topic.substring("persistent://".length());
        }
        if (topic.startsWith("non-persistent://")) {
            return topic.substring("non-persistent://".length());
        }
        return topic;
    }

    boolean forTopic(String topic) {
        boolean nonPersistent = "true".equals(this.claims.get("volatile"));
        String authorizedTopic = (String)this.claims.get("topic");
        if (topic.startsWith("non-persistent://")) {
            return nonPersistent && this.matchesPartitioned(topic, "non-persistent://" + authorizedTopic);
        }
        if (nonPersistent) {
            return false;
        }
        if (topic.startsWith("persistent://")) {
            return this.matchesPartitioned(topic, "persistent://" + authorizedTopic);
        }
        return this.matchesPartitioned(topic, authorizedTopic);
    }

    public boolean isBasicAuth() {
        return Boolean.TRUE.equals(this.claims.get("isBasicAuth"));
    }

    public String getPrinciple() {
        return this.isBasicAuth() ? String.valueOf(this.claims.get("username")) : "N2-token";
    }

    boolean isWriteOnly() {
        return "true".equals(this.claims.get("writeOnly"));
    }

    boolean isReadOnly() {
        return "true".equals(this.claims.get("readOnly"));
    }

    boolean isWebSocketOnly() {
        return "true".equals(this.claims.get("webSocketOnly"));
    }

    private boolean matchesPartitioned(String actualTopic, String authorizedTopic) {
        if (actualTopic.equals(authorizedTopic)) {
            return true;
        }
        if (actualTopic.startsWith(authorizedTopic + "-partition-")) {
            String suffix = actualTopic.substring((authorizedTopic + "-partition-").length());
            return suffix.matches("\\d+");
        }
        return false;
    }

    private static class AuthorizedTopics {
        private final List<AuthorizedTopic> topics;

        @JsonCreator
        public AuthorizedTopics(@JsonProperty(value="topics") List<AuthorizedTopic> topics) {
            this.topics = topics;
        }

        public List<AuthorizedTopic> getTopics() {
            return this.topics;
        }
    }

    private static class AuthorizedTopic {
        private final String name;
        private final List<String> permissions;

        @JsonCreator
        public AuthorizedTopic(@JsonProperty(value="name") String name, @JsonProperty(value="permissions") List<String> permissions) {
            this.name = name;
            this.permissions = permissions;
        }

        public String getName() {
            return this.name;
        }

        public List<String> getPermissions() {
            return this.permissions;
        }
    }
}

