/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.proxy.server;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Getter;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.common.api.AuthData;
import org.apache.pulsar.common.api.proto.CommandConnect;
import org.apache.pulsar.websocket.JwtClaimsHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.*;

import static java.net.http.HttpResponse.BodyHandlers.ofString;
import static java.time.Duration.ofSeconds;
import static javax.ws.rs.core.HttpHeaders.ACCEPT;
import static javax.ws.rs.core.HttpHeaders.AUTHORIZATION;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.Response.Status.OK;

public class Claims {

    private static final Logger LOG = LoggerFactory.getLogger(Claims.class);

    private static final int HTTP_TIMEOUT_SECONDS = 10;
    private static final HttpClient C8Y_HTTP_CLIENT = HttpClient.newBuilder()
            .connectTimeout(Duration.ofSeconds(HTTP_TIMEOUT_SECONDS))
            .build();
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private final Map<String, Object> claims;

    Claims(Map<String, Object> claims) {
        this.claims = claims;
    }

    public static Claims authenticateC8Y(ProxyService service, CommandConnect connect) throws PulsarClientException {
        if (connect.hasAuthMethodName() && "token".equals(connect.getAuthMethodName())) {
            return authenticateC8YToken(service, connect);
        }

        if (connect.hasAuthMethodName() && "basic".equals(connect.getAuthMethodName())) {
            return authenticateC8YBasic(service, connect);
        }

        throw new PulsarClientException.UnsupportedAuthenticationException("Must include a Cumulocity token or use 'basic' authentication method");
    }

    private static Claims authenticateC8YBasic(ProxyService service, CommandConnect connect) throws PulsarClientException {
        final byte[] authData = connect.getAuthData();

        if (ArrayUtils.isEmpty(authData)) {
            throw new PulsarClientException.UnsupportedAuthenticationException(
                    "Authentication failed: Cumulocity basic authentication credentials must not be empty.");
        }

        final String credentials = new String(AuthData.of(authData).getBytes());

        URI baseUri = URI.create(service.getConfiguration().getCumulocityBaseUrl());
        if (baseUri.getScheme() == null) {
            baseUri = URI.create("http://" + baseUri.toString());
        }

        final URI authUri = baseUri.resolve(service.getConfiguration().getCumulocityAuthenticationUrl());

        LOG.debug("Attempting C8Y basic auth. Base URL: <{}>, Auth URL: <{}>",
                service.getConfiguration().getCumulocityBaseUrl(),
                service.getConfiguration().getCumulocityAuthenticationUrl());

        try {
            final HttpRequest request = buildRequest(credentials, authUri);
            final HttpResponse<String> response = C8Y_HTTP_CLIENT.send(request, ofString());

            if (response.statusCode() != OK.getStatusCode()) {
                throw new PulsarClientException.AuthenticationException(
                        "Cumulocity basic authentication failed with status code: " + response.statusCode());
            }

            final AuthorizedTopics parsedResponse = OBJECT_MAPPER.readValue(response.body(), AuthorizedTopics.class);
            final Map<String, Object> claimsMap = new HashMap<>();
            claimsMap.put("isBasicAuth", true);
            claimsMap.put("topics", parsedResponse.getTopics());
            claimsMap.put("username", extractUsername(credentials));

            logAuthEvent(claimsMap);
            return new Claims(claimsMap);
        } catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }

            LOG.error("Error during Cumulocity basic authentication request", e);
            throw new PulsarClientException.AuthenticationException(
                    "A error occurred during basic authentication: " + e.getMessage());
        }
    }

    private static String getAuthenticationString(String credentials) {
        String encoded = Base64.getEncoder().encodeToString(credentials.getBytes(StandardCharsets.UTF_8));
        return "Basic " + encoded;
    }

    private static HttpRequest buildRequest(String credentials, URI uri) {
        return HttpRequest.newBuilder()
                .timeout(ofSeconds(HTTP_TIMEOUT_SECONDS))
                .uri(uri)
                .header(ACCEPT, APPLICATION_JSON)
                .header(AUTHORIZATION, getAuthenticationString(credentials))
                .build();
    }

    public static Claims authenticateC8YToken(ProxyService service, CommandConnect connect) throws PulsarClientException.UnsupportedAuthenticationException {

        if (StringUtils.isEmpty(service.getConfiguration().getProxyPublicKeyFile())) {
            return null;
        }

        if (connect.hasAuthMethodName() && "token".equals(connect.getAuthMethodName())) {
            byte[] authData = connect.getAuthData();
            if (authData == null || authData.length == 0) {
                throw new PulsarClientException.UnsupportedAuthenticationException("Must include a non-empty Cumulocity token");
            }
            AuthData tokenAuthData = AuthData.of(authData);
            String jws = new String(tokenAuthData.getBytes());

            Map<String, Object> claimsMap = JwtClaimsHelper.getJwtClaims(jws, service.getConfiguration().getProxyPublicKeyFile());
            logAuthEvent(claimsMap);

            return new Claims(claimsMap);
        }

        throw new PulsarClientException.UnsupportedAuthenticationException("Must include a Cumulocity token");
    }

    private static String extractUsername(String credentials) {
        final int idx = credentials.indexOf(':');
        return (idx > 0) ? credentials.substring(0, idx) : credentials;
    }

    private static void logAuthEvent(Map<String, Object> claimsMap) {
        if (Boolean.TRUE.equals(claimsMap.get("isBasicAuth"))) {
            LOG.info("C8Y basic authentication successful for user [{}]", claimsMap.get("username"));
            return;
        }

        LOG.info("Proxy connection with C8Y token for topic {} and subscriber {}", claimsMap.get("topic"), claimsMap.get("sub"));
    }

    public boolean hasAnyPermissionFor(String topic) {
        if (isBasicAuth()) {
            // Check if there's any 'read' or 'write' permission for the topic
            return checkBasicAuthPermission(topic, "read") || checkBasicAuthPermission(topic, "write");
        }

        // For JWT, the original check is sufficient
        return forTopic(topic);
    }

    public boolean canProduce(String topic) {
        if (isBasicAuth()) {
            return checkBasicAuthPermission(topic, "write");
        }

        return forTopic(topic) && !isReadOnly();
    }

    public boolean canConsume(String topic) {
        if (isBasicAuth()) {
            return checkBasicAuthPermission(topic, "read");
        }

        return forTopic(topic) && !isWriteOnly();
    }

    private boolean checkBasicAuthPermission(String requestedTopic, String requiredPermission) {
        @SuppressWarnings("unchecked") final List<AuthorizedTopic> topics = (List<AuthorizedTopic>) claims.get("topics");

        if (topics == null || topics.isEmpty()) {
            return false;
        }

        final String cleanTopicName = cleanTopicName(requestedTopic);

        return topics.stream()
                .filter(authorizedTopic -> matchesPartitioned(cleanTopicName, authorizedTopic.getName()))
                .map(AuthorizedTopic::getPermissions)
                .filter(Objects::nonNull)
                .anyMatch(permissions -> permissions.contains(requiredPermission));
    }

    private static String cleanTopicName(String topic) {
        if (topic.startsWith("persistent://")) {
            return topic.substring("persistent://".length());
        }

        if (topic.startsWith("non-persistent://")) {
            return topic.substring("non-persistent://".length());
        }

        return topic;
    }

    boolean forTopic(String topic) {
        boolean nonPersistent = "true".equals(claims.get("volatile"));
        String authorizedTopic = (String) claims.get("topic");

        if (topic.startsWith("non-persistent://")) {
            return nonPersistent && matchesPartitioned(topic, "non-persistent://" + authorizedTopic);
        }
        if (nonPersistent) {
            return false;
        }
        if (topic.startsWith("persistent://")) {
            return matchesPartitioned(topic, "persistent://" + authorizedTopic);
        }
        return matchesPartitioned(topic, authorizedTopic);
    }

    public boolean isBasicAuth() {
        return Boolean.TRUE.equals(claims.get("isBasicAuth"));
    }

    public String getPrinciple() {
        return isBasicAuth() ? String.valueOf(claims.get("username")) : "N2-token";
    }

    boolean isWriteOnly() {
        return "true".equals(claims.get("writeOnly"));
    }

    boolean isReadOnly() {
        return "true".equals(claims.get("readOnly"));
    }

    boolean isWebSocketOnly() {
        return "true".equals(claims.get("webSocketOnly"));
    }

    private boolean matchesPartitioned(String actualTopic, String authorizedTopic) {
        // exact match
        if (actualTopic.equals(authorizedTopic)) {
            return true;
        }

        // check partitioned form: authorized + "-partition-" + <number>
        if (actualTopic.startsWith(authorizedTopic + "-partition-")) {
            final String suffix = actualTopic.substring((authorizedTopic + "-partition-").length());
            return suffix.matches("\\d+");  // must be all digits
        }

        return false;
    }

    @Getter
    private static class AuthorizedTopics {
        private final List<AuthorizedTopic> topics;

        @JsonCreator
        public AuthorizedTopics(@JsonProperty("topics") List<AuthorizedTopic> topics) {
            this.topics = topics;
        }
    }

    @Getter
    private static class AuthorizedTopic {
        private final String name;
        private final List<String> permissions;

        @JsonCreator
        public AuthorizedTopic(
                @JsonProperty("name") String name,
                @JsonProperty("permissions") List<String> permissions) {
            this.name = name;
            this.permissions = permissions;
        }
    }
}
