/*
 * Decompiled with CFR 0.152.
 */
package org.svenson.tokenize;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.svenson.JSONParseException;
import org.svenson.tokenize.JSONCharacterSource;
import org.svenson.tokenize.StringJSONSource;
import org.svenson.tokenize.Token;
import org.svenson.tokenize.TokenType;

public class JSONTokenizer {
    private static Logger log = LoggerFactory.getLogger(JSONTokenizer.class);
    private JSONCharacterSource source;
    private boolean isDecimal;
    private char pushedBack;
    private List<Token> recordedTokens = new ArrayList<Token>();
    private boolean allowSingleQuotes = false;
    private boolean recording;
    private boolean tokenPushedBack;
    private boolean reachedEndOfJSON;
    private int pushBackIndex;
    private static final int HEX_LETTER_OFFSET = 7;

    public JSONTokenizer(String json, boolean allowSingleQuotes) {
        if (json == null) {
            throw new IllegalArgumentException("json string cannot be null.");
        }
        this.source = new StringJSONSource(json);
        this.allowSingleQuotes = allowSingleQuotes;
    }

    public JSONTokenizer(JSONCharacterSource source, boolean allowSingleQuotes) {
        if (source == null) {
            throw new IllegalArgumentException("character source cannot be null.");
        }
        this.source = source;
        this.allowSingleQuotes = allowSingleQuotes;
    }

    public void destroy() {
        this.source.destroy();
    }

    public boolean isAllowSingleQuotes() {
        return this.allowSingleQuotes;
    }

    private boolean isNumberCharacter(char c) {
        switch (c) {
            case '+': 
            case '-': 
            case '.': 
            case 'E': 
            case 'e': {
                this.isDecimal = true;
                return true;
            }
            case '0': 
            case '1': 
            case '2': 
            case '3': 
            case '4': 
            case '5': 
            case '6': 
            case '7': 
            case '8': 
            case '9': {
                return true;
            }
        }
        return false;
    }

    private void ensureKeywordSuffix(String word) {
        String suffix = word.substring(1);
        int length = suffix.length();
        for (int i = 0; i < length; ++i) {
            if (this.nextChar() == suffix.charAt(i)) continue;
            throw new JSONParseException("invalid keyword at index " + this.source.getIndex() + " (should be '" + word + "')");
        }
    }

    public final Token next() {
        Token token;
        if (this.tokenPushedBack) {
            Token token2;
            if (this.recording) {
                token2 = this.recordedTokens.get(this.pushBackIndex++);
                if (this.recordedTokens.size() == this.pushBackIndex) {
                    this.tokenPushedBack = false;
                }
            } else {
                token2 = this.recordedTokens.remove(0);
                if (this.recordedTokens.size() == 0) {
                    this.tokenPushedBack = false;
                }
            }
            log.trace("token = {}", (Object)token2);
            return token2;
        }
        this.skipWhiteSpace();
        if (this.reachedEndOfJSON && this.pushedBack == '\u0000') {
            return Token.getToken(TokenType.END);
        }
        this.isDecimal = false;
        int c1 = this.nextChar();
        switch ((char)c1) {
            case '\"': {
                token = this.parseString((char)c1);
                break;
            }
            case '[': {
                token = Token.getToken(TokenType.BRACKET_OPEN, "[");
                break;
            }
            case ']': {
                token = Token.getToken(TokenType.BRACKET_CLOSE, "]");
                break;
            }
            case '{': {
                token = Token.getToken(TokenType.BRACE_OPEN, "{");
                break;
            }
            case '}': {
                token = Token.getToken(TokenType.BRACE_CLOSE, "}");
                break;
            }
            case ':': {
                token = Token.getToken(TokenType.COLON, ":");
                break;
            }
            case ',': {
                token = Token.getToken(TokenType.COMMA, ",");
                break;
            }
            case 't': {
                this.ensureKeywordSuffix("true");
                token = Token.getToken(TokenType.TRUE, Boolean.TRUE);
                break;
            }
            case 'f': {
                this.ensureKeywordSuffix("false");
                token = Token.getToken(TokenType.FALSE, Boolean.FALSE);
                break;
            }
            case 'n': {
                this.ensureKeywordSuffix("null");
                token = Token.getToken(TokenType.NULL);
                break;
            }
            default: {
                if (this.isNumberCharacter((char)c1)) {
                    token = this.parseNumber((char)c1);
                    break;
                }
                if (c1 == 39 && this.allowSingleQuotes) {
                    token = this.parseString((char)c1);
                    break;
                }
                throw new JSONParseException("Unexpected character '" + (char)c1 + "'");
            }
        }
        if (this.recording) {
            this.recordedTokens.add(token);
        }
        log.trace("token = {}", (Object)token);
        return token;
    }

    public void pushBack(Token oldToken) {
        int index = this.recordedTokens.indexOf(oldToken);
        if (index < 0) {
            throw new IllegalStateException("Can't rollback to non-recorded token " + oldToken);
        }
        this.pushBackToIndex(index, oldToken);
    }

    public void pushBackAfterOrToFirstRecorded(Token oldToken) {
        int index = this.recordedTokens.indexOf(oldToken);
        if (index < 0 && this.recordedTokens.isEmpty()) {
            throw new IllegalStateException("Can't rollback after non-recorded token " + oldToken);
        }
        this.pushBackToIndex(index + 1, this.recordedTokens.get(index + 1));
    }

    private void pushBackToIndex(int index, Token oldToken) {
        if (index < 0) {
            throw new IllegalStateException("Can't rollback to non-recorded token " + oldToken);
        }
        if (index > 0) {
            this.recordedTokens = this.recordedTokens.subList(index, this.recordedTokens.size());
        }
        this.tokenPushedBack = true;
        this.recording = false;
        this.pushBackIndex = 0;
    }

    private Token parseNumber(char c1) {
        int c;
        if (c1 == '-') {
            this.isDecimal = false;
        }
        StringBuilder sb = new StringBuilder();
        sb.append(c1);
        while ((c = this.nextChar()) > -1) {
            if (!this.isNumberCharacter((char)c)) {
                this.pushBack((char)c);
                break;
            }
            sb.append((char)c);
        }
        String number = sb.toString();
        if (this.isDecimal) {
            return this.parseDecimal(number);
        }
        try {
            long l = Long.parseLong(number);
            return Token.getToken(TokenType.INTEGER, l);
        }
        catch (NumberFormatException nfe) {
            return this.parseDecimal(number);
        }
    }

    private void pushBack(char c) {
        this.pushedBack = c;
    }

    private Token parseDecimal(String number) {
        try {
            BigDecimal d = new BigDecimal(number);
            if (number.contains("e") || number.contains("E")) {
                return Token.getToken(TokenType.DECIMAL, d.doubleValue());
            }
            return Token.getToken(TokenType.DECIMAL, d);
        }
        catch (NumberFormatException nfe) {
            throw new JSONParseException("Error parsing decimal " + number);
        }
    }

    private Token parseString(char quoteChar) {
        int c;
        StringBuilder sb = new StringBuilder();
        boolean escape = false;
        while ((c = this.nextChar()) >= 0) {
            if (c == quoteChar && !escape) {
                return Token.getToken(TokenType.STRING, sb.toString());
            }
            if (c == 92) {
                if (escape) {
                    sb.append('\\');
                }
                escape = !escape;
                continue;
            }
            if (escape) {
                switch ((char)c) {
                    case '\"': 
                    case '\'': 
                    case '/': {
                        sb.append((char)c);
                        break;
                    }
                    case 'b': {
                        sb.append('\b');
                        break;
                    }
                    case 'f': {
                        sb.append('\f');
                        break;
                    }
                    case 'n': {
                        sb.append('\n');
                        break;
                    }
                    case 'r': {
                        sb.append('\r');
                        break;
                    }
                    case 't': {
                        sb.append('\t');
                        break;
                    }
                    case 'u': {
                        int unicode = (JSONTokenizer.hexValue((char)this.nextChar()) << 12) + (JSONTokenizer.hexValue((char)this.nextChar()) << 8) + (JSONTokenizer.hexValue((char)this.nextChar()) << 4) + JSONTokenizer.hexValue((char)this.nextChar());
                        sb.append((char)unicode);
                        break;
                    }
                    default: {
                        throw new JSONParseException("Illegal escape character " + c + " / " + Integer.toHexString(c));
                    }
                }
                escape = false;
                continue;
            }
            if (Character.isISOControl(c)) {
                throw new JSONParseException("Illegal control character 0x" + Integer.toHexString(c));
            }
            sb.append((char)c);
        }
        throw new JSONParseException("Unclosed quotes");
    }

    public static int hexValue(char c) {
        int n = c;
        if (n >= 97) {
            n &= 0xFFFFFFDF;
        }
        if (n >= 48 && n <= 57 || n >= 65 && n <= 70) {
            if ((n -= 48) > 9) {
                return n - 7;
            }
            return n;
        }
        throw new NumberFormatException("Invalid hex character " + (char)c);
    }

    private String info() {
        return "at character offset " + this.source.getIndex();
    }

    private boolean isCR(char c) {
        return c == '\r' || c == '\n';
    }

    private int nextChar() {
        if (this.pushedBack != '\u0000') {
            char c = this.pushedBack;
            this.pushedBack = '\u0000';
            return c;
        }
        int c = this.source.nextChar();
        if (c < 0) {
            this.reachedEndOfJSON = true;
        }
        return c;
    }

    private void skipWhiteSpace() {
        int c;
        block3: while ((c = this.nextChar()) != -1) {
            switch (c) {
                case 8: 
                case 9: 
                case 10: 
                case 13: 
                case 32: {
                    continue block3;
                }
            }
            this.pushBack((char)c);
            return;
        }
    }

    public Token expectNext(TokenType ... types) {
        Token t = this.next();
        t.expect(types);
        return t;
    }

    public void startRecording() {
        this.recording = true;
    }

    public Token peekToken() {
        boolean wasRecording = this.recording;
        this.startRecording();
        Token token = this.next();
        this.pushBack(token);
        if (wasRecording) {
            this.startRecording();
        }
        return token;
    }
}

