/*
 * Decompiled with CFR 0.152.
 */
package com.cumulocity.opcua.client.gateway.mappingsexecution;

import com.cumulocity.opcua.client.gateway.jmx.CustomActionMBean;
import com.cumulocity.opcua.client.gateway.mappingsexecution.HttpPostElementComparator;
import com.cumulocity.opcua.client.gateway.mappingsexecution.model.HttpPostElement;
import com.cumulocity.opcua.client.gateway.mappingsexecution.tasks.MappingExecutionTaskFactory;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.task.TaskRejectedException;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Component;

@Component
public class HttpPostQueue {
    private static final Logger log = LoggerFactory.getLogger(HttpPostQueue.class);
    private final boolean retryEnabled;
    private final Set<Long> noRetryHttpCodes;
    private final int retryDelay;
    private final int maxRetries;
    private final int queueBufferMaxSize;
    private final ThreadPoolTaskScheduler scheduler;
    private final ThreadPoolTaskExecutor executor;
    private final MappingExecutionTaskFactory factory;
    private final ConcurrentSkipListSet<HttpPostElement> actions;
    private final AtomicLong elementSequence;
    private CustomActionMBean customActionMBean;
    private AtomicLong queueSize;

    @Autowired
    public HttpPostQueue(@Value(value="${gateway.mappingExecution.http.failureHandling.enabled:false}") boolean retryEnabled, @Value(value="${gateway.mappingExecution.http.failureHandling.noRetryHttpCodes:}") Set<Long> noRetryHttpCodes, @Value(value="${gateway.mappingExecution.http.failureHandling.retryDelay:120}") int retryDelay, @Value(value="${gateway.mappingExecution.http.failureHandling.maxRetries:5}") int maxRetries, @Value(value="${gateway.mappingExecution.http.maxQueueSize:50000}") int maxQueueSize, ThreadPoolTaskScheduler scheduler, @Qualifier(value="httpPostTaskExecutor") ThreadPoolTaskExecutor executor, MappingExecutionTaskFactory factory, CustomActionMBean customActionMBean) {
        this.retryEnabled = retryEnabled;
        this.noRetryHttpCodes = noRetryHttpCodes;
        this.retryDelay = retryDelay;
        this.maxRetries = maxRetries;
        this.queueBufferMaxSize = maxQueueSize / 2;
        this.scheduler = scheduler;
        this.executor = executor;
        this.factory = factory;
        this.queueSize = new AtomicLong(0L);
        this.elementSequence = new AtomicLong(0L);
        this.actions = new ConcurrentSkipListSet(new HttpPostElementComparator());
        this.customActionMBean = customActionMBean;
    }

    public void afterPropertiesSet() {
        this.scheduler.scheduleWithFixedDelay(() -> this.ensureMaxSize(), Duration.of(10L, ChronoUnit.SECONDS).toMillis());
        log.info("Scheduled housekeeping job to ensure queue max size of the queue - every 10 seconds");
        this.scheduler.scheduleWithFixedDelay(() -> this.flush(), Duration.of(100L, ChronoUnit.MILLIS).toMillis());
        log.info("Scheduled queue flushing to run every 100 milliseconds");
    }

    public void add(HttpPostElement element) {
        boolean added = this.actions.add(element);
        if (added) {
            this.queueSize.incrementAndGet();
        }
    }

    void remove(HttpPostElement element) {
        boolean removed = this.actions.remove(element);
        if (removed) {
            this.queueSize.decrementAndGet();
        }
    }

    public void onFailed(HttpPostElement action, long statusCode) {
        boolean retryEnabled;
        log.warn("Http post to endpoint: {} failed with status code: {}", (Object)action.getEndpoint(), (Object)statusCode);
        log.debug("Http post element: {}", (Object)action);
        boolean bl = retryEnabled = Objects.nonNull(action.getRetryEnabled()) && action.getRetryEnabled() != false || Objects.isNull(action.getRetryEnabled()) && this.retryEnabled;
        if (!retryEnabled) {
            log.warn("Retry disabled, no retry as per configuration, discarding element: {}", (Object)action);
            return;
        }
        Set noRetryHttpCodes = this.noRetryHttpCodes;
        if (Objects.nonNull(action.getNoRetryHttpCodes())) {
            noRetryHttpCodes = action.getNoRetryHttpCodes();
        }
        if (noRetryHttpCodes.contains(statusCode)) {
            log.warn("Status code: {}, no retry as per configuration, discarding element: {}", (Object)statusCode, (Object)action);
            return;
        }
        this.remove(action);
        if (!action.markAsRetried(this.maxRetries)) {
            log.warn("Max retries reached, discarding element: {}", (Object)action);
        } else {
            log.info("Adding http post element back to the queue");
            this.add(action);
        }
    }

    public long nextElementId() {
        if (this.elementSequence.get() == Long.MAX_VALUE) {
            this.elementSequence.set(0L);
        }
        return this.elementSequence.incrementAndGet();
    }

    public long getQueueSize() {
        return this.queueSize.get();
    }

    void ensureMaxSize() {
        long size = this.queueSize.get();
        log.info("Queue buffer size: {}/{}", (Object)size, (Object)this.queueBufferMaxSize);
        if (size > (long)this.queueBufferMaxSize) {
            log.info("Queue buffer is full, removing elements from the queue");
        }
        int removed = 0;
        while (size > (long)this.queueBufferMaxSize) {
            HttpPostElement element = this.doPollLast();
            if (!Objects.nonNull(element)) continue;
            log.warn("Discarded element: {}", (Object)element);
            size = this.queueSize.get();
            ++removed;
        }
        if (removed > 0) {
            log.info("Removed {} elements from the queue", (Object)removed);
        }
        this.customActionMBean.setRetryQueueSize(size);
    }

    void flush() {
        HttpPostElement element = this.pollFirst();
        while (Objects.nonNull(element)) {
            try {
                this.executor.submit(this.factory.createHttpPostTask(element));
            }
            catch (TaskRejectedException exception) {
                log.warn("Http post task rejected, message: {}. Most probably the thread queue is full, adding back to http post element queue", (Object)exception.getMessage());
                this.add(element);
                try {
                    Thread.sleep(Duration.ofSeconds(5L).toMillis());
                }
                catch (InterruptedException e) {
                    log.error("Queue flushing interrupted");
                    Thread.currentThread().interrupt();
                }
            }
            element = this.pollFirst();
        }
    }

    private HttpPostElement pollFirst() {
        if (this.actions.isEmpty()) {
            return null;
        }
        HttpPostElement action = (HttpPostElement)this.actions.first();
        if (Objects.nonNull(action) && !action.shouldTryNow(this.retryDelay)) {
            return null;
        }
        return this.doPollFirst();
    }

    private HttpPostElement doPollFirst() {
        HttpPostElement element = (HttpPostElement)this.actions.pollFirst();
        if (Objects.nonNull(element)) {
            this.queueSize.decrementAndGet();
        }
        return element;
    }

    private HttpPostElement doPollLast() {
        HttpPostElement element = (HttpPostElement)this.actions.pollLast();
        if (Objects.nonNull(element)) {
            this.queueSize.decrementAndGet();
        }
        return element;
    }

    public int getQueueBufferMaxSize() {
        return this.queueBufferMaxSize;
    }
}

