/*
 * Decompiled with CFR 0.152.
 */
package com.cumulocity.opcua.client.gateway.monitoring;

import com.cumulocity.model.event.CumulocityAlarmStatuses;
import com.cumulocity.model.idtype.GId;
import com.cumulocity.opcua.client.OpcuaClient;
import com.cumulocity.opcua.client.exception.OpcuaClientException;
import com.cumulocity.opcua.client.gateway.GatewayManager;
import com.cumulocity.opcua.client.gateway.ServerIdentifier;
import com.cumulocity.opcua.client.gateway.connection.ConnectionManager;
import com.cumulocity.opcua.client.gateway.connection.model.ConnectionEstablishedEvent;
import com.cumulocity.opcua.client.gateway.connection.model.ReestablishConnectionEvent;
import com.cumulocity.opcua.client.gateway.connection.model.ServerConnectedEvent;
import com.cumulocity.opcua.client.gateway.connection.model.ServerConnectionDroppedEvent;
import com.cumulocity.opcua.client.gateway.connection.model.ServerConnectionFailedEvent;
import com.cumulocity.opcua.client.gateway.connection.model.ServerDisconnectedEvent;
import com.cumulocity.opcua.client.gateway.jmx.ServerMonitoringMBean;
import com.cumulocity.opcua.client.gateway.platform.repository.AlarmRepository;
import com.cumulocity.opcua.client.gateway.platform.repository.EventRepository;
import com.cumulocity.opcua.common.repository.InventoryRepository;
import com.cumulocity.rest.representation.inventory.ManagedObjectRepresentation;
import com.cumulocity.sdk.client.QueryParam;
import com.cumulocity.sdk.client.SDKException;
import com.cumulocity.sdk.client.alarm.AlarmFilter;
import com.cumulocity.sdk.client.alarm.PagedAlarmCollectionRepresentation;
import com.prosysopc.ua.stack.builtintypes.NodeId;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

@Service
public class ConnectionMonitoringService {
    private static final Logger log = LoggerFactory.getLogger(ConnectionMonitoringService.class);
    private static final String UA_SERVER_CONNECTION_FAILED_ALARM_SEVERITY = "CRITICAL";
    private static final int CONNECTION_DROPPED_ALARM_BREAK_DURATION_MINUTES = 60;
    @Autowired
    private ConnectionManager connectionManager;
    @Autowired
    private ApplicationEventPublisher eventPublisher;
    @Autowired
    private EventRepository eventRepository;
    @Autowired
    private AlarmRepository alarmRepository;
    @Autowired
    private GatewayManager gatewayManager;
    @Autowired
    private InventoryRepository inventoryRepository;
    @Autowired
    private ServerMonitoringMBean mBean;
    @Value(value="${gateway.monitoring.interval:20000}")
    private int hardCheckInterval;
    final Map<String, NodeId> activeSessions = new ConcurrentHashMap();
    final Map<String, DateTime> hardCheckDateTimes = new ConcurrentHashMap();
    private final Map<String, BigDecimal> responseTimes = new ConcurrentHashMap();

    @Scheduled(fixedDelayString="${gateway.connectionMonitoring.fixedDelay:3000}")
    public void monitor() {
        Collection allServers = this.connectionManager.getAllServerIdentifiers();
        for (ServerIdentifier serverId : allServers) {
            Optional connection = this.connectionManager.getConnection(serverId);
            connection.ifPresent(client -> {
                boolean noChangesDetected = this.monitorSession(serverId, client);
                if (noChangesDetected) {
                    this.doHardCheckIfNeeded(serverId, client);
                }
            });
        }
    }

    @EventListener
    public void onServerConnected(ServerConnectedEvent event) {
        try {
            this.eventRepository.createEvent(event.getServerIdentifier().getInventoryIdentifier(), "c8y_ua_ServerConnected", String.format("Server [%s] connected", event.getServerIdentifier().getInventoryIdentifier().getValue()));
            AlarmFilter connectionFailedFilter = new AlarmFilter().byType("c8y_ua_ServerConnectionFailed").bySource(event.getServerIdentifier().getInventoryIdentifier()).byStatus(new CumulocityAlarmStatuses[]{CumulocityAlarmStatuses.ACTIVE}).byResolved(Boolean.valueOf(false));
            this.alarmRepository.clearAlarmsByFilter(connectionFailedFilter);
        }
        catch (Exception e) {
            log.error("Unable to handle server connected event, monitoring events/alarms might not be created or cleared", (Throwable)e);
        }
    }

    @EventListener
    public void onServerConnectionFailed(ServerConnectionFailedEvent event) {
        try {
            this.mBean.markServerDisconnected(event.getServerIdentifier().getName());
            this.alarmRepository.create(event.getServerIdentifier().getInventoryIdentifier(), "c8y_ua_ServerConnectionFailed", UA_SERVER_CONNECTION_FAILED_ALARM_SEVERITY, String.format("Failed to connect to server [%s], reason: %s", event.getServerIdentifier().getInventoryIdentifier().getValue(), event.getMessage()));
        }
        catch (Exception e) {
            log.error("Unable to create connection failed alarm, server: {}", (Object)event.getServerIdentifier().getInventoryIdentifier(), (Object)e);
        }
    }

    @EventListener
    public void onServerDisconnected(ServerDisconnectedEvent event) {
        try {
            this.mBean.markServerDisconnected(event.getServerIdentifier().getName());
            this.createServerDisconnectedEvent(event);
        }
        catch (Exception e) {
            log.error("Unable to create event on server disconnection", (Throwable)e);
        }
    }

    @EventListener
    public void onConnectionEstablished(ConnectionEstablishedEvent event) {
        try {
            this.createConnectionEstablishedEvent(event.getServerId());
            this.clearConnectionDroppedAlarm(event.getServerId());
        }
        finally {
            if (Objects.nonNull(event.getSessionId())) {
                this.activeSessions.put(event.getServerId(), event.getSessionId());
            }
        }
    }

    private boolean monitorSession(ServerIdentifier serverId, OpcuaClient connection) {
        Optional sessionIdMaybe = connection.getSessionId();
        String serverIdString = serverId.getInventoryIdentifier().getValue();
        if (log.isTraceEnabled()) {
            log.trace("Old session: {}, new session: {}", this.activeSessions.get(serverIdString), sessionIdMaybe.orElse(null));
        }
        if (sessionIdMaybe.isPresent()) {
            this.mBean.markServerConnected(serverId.getName());
        }
        if (!sessionIdMaybe.isPresent()) {
            this.mBean.markServerDisconnected(serverId.getName());
            this.handleSessionMissing(serverIdString);
            return false;
        }
        if (!((NodeId)sessionIdMaybe.get()).equals(this.activeSessions.get(serverIdString))) {
            this.handleSessionChanged(serverId.getInventoryIdentifier().getValue(), (NodeId)sessionIdMaybe.get());
            this.eventPublisher.publishEvent((Object)new ReestablishConnectionEvent(serverId));
            return false;
        }
        return true;
    }

    private void handleSessionMissing(String serverIdString) {
        log.warn("Connection session to server {} is missing", (Object)serverIdString);
        this.createConnectionDroppedAlarm(serverIdString);
        this.eventPublisher.publishEvent((Object)new ServerConnectionDroppedEvent(serverIdString));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleSessionChanged(String serverIdString, NodeId newSession) {
        try {
            NodeId oldSessionId = (NodeId)this.activeSessions.get(serverIdString);
            log.info("Established connection to server: {} session ID : {} -> {}", new Object[]{serverIdString, oldSessionId, newSession});
            this.eventPublisher.publishEvent((Object)new ConnectionEstablishedEvent(serverIdString, newSession, Objects.nonNull(oldSessionId)));
        }
        finally {
            this.activeSessions.put(serverIdString, newSession);
        }
    }

    private void doHardCheckIfNeeded(ServerIdentifier serverIdentifier, OpcuaClient connection) {
        boolean timeToCheck;
        String serverId = serverIdentifier.getInventoryIdentifier().getValue();
        DateTime lastHardCheck = (DateTime)this.hardCheckDateTimes.get(serverId);
        boolean bl = timeToCheck = Objects.isNull(lastHardCheck) || lastHardCheck.plusMillis(this.hardCheckInterval).isBefore((ReadableInstant)DateTime.now());
        if (!timeToCheck) {
            return;
        }
        this.hardCheckDateTimes.put(serverId, DateTime.now());
        long start = System.currentTimeMillis();
        try {
            connection.testConnection();
            long end = System.currentTimeMillis();
            this.responseTimes.put(serverId, new BigDecimal(end - start));
            this.mBean.markServerConnected(serverIdentifier.getName());
        }
        catch (OpcuaClientException e) {
            log.warn("Test connection failed on server: {}, publishing event", (Object)serverId, (Object)e);
            this.createConnectionDroppedAlarm(serverId);
            this.eventPublisher.publishEvent((Object)new ServerConnectionDroppedEvent(serverId));
            this.mBean.markServerDisconnected(serverIdentifier.getName());
        }
    }

    BigDecimal takeServerResponseTime(String serverId) {
        return (BigDecimal)this.responseTimes.remove(serverId);
    }

    private void createConnectionEstablishedEvent(String serverId) {
        try {
            this.eventRepository.createEvent(GId.asGId((String)serverId), "c8y_ua_ConnectionEstablished", "Connection established to server: " + serverId);
        }
        catch (Exception e) {
            log.error("Unable to create event on connection established, server: {}", (Object)serverId, (Object)e);
        }
    }

    private void createConnectionDroppedAlarm(String serverId) {
        try {
            AlarmFilter alarmFilter = new AlarmFilter().byStatus(new CumulocityAlarmStatuses[]{CumulocityAlarmStatuses.ACTIVE}).bySource(GId.asGId((String)serverId)).byType("c8y_ua_ConnectionDropped").byResolved(Boolean.valueOf(false)).byFromDate(DateTime.now().minusMinutes(60).toDate());
            List activeAlarms = null;
            try {
                activeAlarms = ((PagedAlarmCollectionRepresentation)this.alarmRepository.getAlarmsByFilter(alarmFilter).get(1, new QueryParam[0])).getAlarms();
            }
            catch (Exception e) {
                log.warn("Unable to get ACTIVE connection dropped alarm", (Throwable)e);
            }
            if (CollectionUtils.isEmpty(activeAlarms)) {
                this.alarmRepository.create(GId.asGId((String)serverId), "c8y_ua_ConnectionDropped", UA_SERVER_CONNECTION_FAILED_ALARM_SEVERITY, "Connection dropped on server: " + serverId);
            } else if (log.isDebugEnabled()) {
                log.debug("Connection dropped alarm is still ACTIVE, don't create another one.");
            }
        }
        catch (Exception e) {
            log.error("Unable to create alarm on connection dropped, server: {}", (Object)serverId, (Object)e);
        }
    }

    private void clearConnectionDroppedAlarm(String serverId) {
        AlarmFilter alarmFilter = new AlarmFilter().byStatus(new CumulocityAlarmStatuses[]{CumulocityAlarmStatuses.ACTIVE}).bySource(GId.asGId((String)serverId)).byType("c8y_ua_ConnectionDropped").byResolved(Boolean.valueOf(false));
        try {
            this.alarmRepository.clearAlarmsByFilter(alarmFilter);
        }
        catch (Exception e) {
            log.warn("Unable to clear connection dropped alarm, server: {}", (Object)serverId, (Object)e);
        }
    }

    private void createServerDisconnectedEvent(ServerDisconnectedEvent event) {
        GId eventSource = event.getServerIdentifier().getInventoryIdentifier();
        try {
            ManagedObjectRepresentation serverMO = this.inventoryRepository.get(eventSource);
            if (Objects.isNull(serverMO)) {
                eventSource = this.gatewayManager.getGatewayDetails().getGatewayDevice().getId();
            }
        }
        catch (SDKException e) {
            if (e.getHttpStatus() != 404) {
                log.warn("Unable to get serer managed object for server: {}", (Object)eventSource.getValue(), (Object)e);
            }
            eventSource = this.gatewayManager.getGatewayDetails().getGatewayDevice().getId();
        }
        this.eventRepository.createEvent(eventSource, "c8y_ua_ServerDisconnected", String.format("Server [%s] disconnected", event.getServerIdentifier()));
    }
}

