/*
 * Decompiled with CFR 0.152.
 */
package com.cumulocity.opcua.client.gateway.operation;

import c8y.ua.command.BaseOperation;
import com.cumulocity.model.idtype.GId;
import com.cumulocity.model.operation.OperationStatus;
import com.cumulocity.opcua.client.gateway.configuration.GatewayGeneralConfiguration;
import com.cumulocity.opcua.client.gateway.notification.OperationSubscriber;
import com.cumulocity.opcua.client.gateway.operation.handler.base.OperationHandler;
import com.cumulocity.rest.representation.operation.OperationRepresentation;
import com.cumulocity.sdk.client.devicecontrol.DeviceControlApi;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import net.jodah.expiringmap.ExpiringMap;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.svenson.SvensonRuntimeException;

@Component
public class OperationExecutor
implements InitializingBean {
    private static final Logger log = LoggerFactory.getLogger(OperationExecutor.class);
    private static final long EXECUTION_DELAY = 100L;
    @Autowired
    private GatewayGeneralConfiguration generalConfiguration;
    @Autowired
    private DeviceControlApi deviceControlApi;
    @Autowired
    private ApplicationContext ctx;
    @Autowired
    private ThreadPoolTaskExecutor executor;
    private final ExpiringMap<String, DateTime> executedOperationsCache = ExpiringMap.builder().expiration(1L, TimeUnit.DAYS).build();
    private Collection<OperationHandler<?>> operationHandlers = new ArrayList();
    private Queue<OperationRepresentation> pendingOperations;
    private Map<String, Object> locks = new ConcurrentHashMap();
    private Future<?> executorJob;

    public void submit(OperationRepresentation operationRepresentation) {
        try {
            if (!this.isOperationInTheQueue(operationRepresentation)) {
                this.pendingOperations.add(operationRepresentation);
            } else if (log.isDebugEnabled()) {
                log.debug("Operation {} has been already in the queue", (Object)operationRepresentation.getId().getValue());
            }
        }
        catch (IllegalStateException ex) {
            log.warn("Operations queue is full, rejecting operation: {}", (Object)operationRepresentation);
            this.setOperationFailedWithReason(operationRepresentation.getId(), "Agent operation queue is full!");
        }
    }

    @Scheduled(fixedDelay=60000L)
    public void monitorExecutor() {
        if (this.executorJob.isDone() || this.executorJob.isCancelled()) {
            this.startOperationExecutor();
        }
    }

    @PostConstruct
    private void startOperationExecutor() {
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        this.executorJob = executorService.submit(() -> {
            while (true) {
                this.execute();
                try {
                    Thread.sleep(100L);
                }
                catch (InterruptedException e) {
                    log.error("Operation executor job interrupted", (Throwable)e);
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        });
    }

    void execute() {
        while (!this.pendingOperations.isEmpty()) {
            Object lock;
            OperationRepresentation polledOperation = (OperationRepresentation)this.pendingOperations.poll();
            if (this.locks.containsKey(polledOperation.getId().getValue())) {
                lock = this.locks.get(polledOperation.getId().getValue());
            } else {
                lock = new Object();
                this.locks.put(polledOperation.getId().getValue(), lock);
            }
            this.executor.execute(() -> {
                Object object = lock;
                synchronized (object) {
                    try {
                        OperationRepresentation operation = polledOperation;
                        GId operationId = operation.getId();
                        if (this.isOperationOutdated(operation)) {
                            try {
                                operation = this.deviceControlApi.getOperation(operation.getId());
                            }
                            catch (SvensonRuntimeException e) {
                                log.warn("Error parsing operation with Svenson - setting operation " + operationId.getValue() + " as failed");
                                this.setOperationFailedWithReason(operationId, "Error parsing operation JSON! This operation is not supported.");
                                this.resetLongPollingSession();
                                this.locks.remove(polledOperation.getId().getValue());
                                return;
                            }
                        }
                        if (!OperationStatus.PENDING.name().equalsIgnoreCase(operation.getStatus())) {
                            log.info("Operation {} is not in PENDING state, skip executing.", (Object)operation.getId());
                            return;
                        }
                        long start = System.currentTimeMillis();
                        if (this.executedOperationsCache.containsKey((Object)operation.getId().getValue())) {
                            log.info("Operation has been executed, skipping: {}", (Object)operation.getId());
                            return;
                        }
                        try {
                            log.info("Start executing operation: [{}]", (Object)operation.getId().getValue());
                            operation.setStatus(OperationStatus.EXECUTING.name());
                            this.deviceControlApi.update(operation);
                            this.executeOperation(operation);
                        }
                        catch (Throwable throwable) {
                            long millisElapsed = System.currentTimeMillis() - start;
                            long minutesElapsed = TimeUnit.MILLISECONDS.toMinutes(millisElapsed);
                            String humanReadableTimeElapsed = String.format("%d min, %d seconds", minutesElapsed, TimeUnit.MILLISECONDS.toSeconds(millisElapsed) - TimeUnit.MINUTES.toSeconds(minutesElapsed));
                            log.info("Operation [{}] execution has been done, time elapsed: {}", (Object)operation.getId().getValue(), (Object)humanReadableTimeElapsed);
                            this.executedOperationsCache.put((Object)operation.getId().getValue(), (Object)DateTime.now());
                            throw throwable;
                        }
                        long millisElapsed = System.currentTimeMillis() - start;
                        long minutesElapsed = TimeUnit.MILLISECONDS.toMinutes(millisElapsed);
                        String humanReadableTimeElapsed = String.format("%d min, %d seconds", minutesElapsed, TimeUnit.MILLISECONDS.toSeconds(millisElapsed) - TimeUnit.MINUTES.toSeconds(minutesElapsed));
                        log.info("Operation [{}] execution has been done, time elapsed: {}", (Object)operation.getId().getValue(), (Object)humanReadableTimeElapsed);
                        this.executedOperationsCache.put((Object)operation.getId().getValue(), (Object)DateTime.now());
                    }
                    finally {
                        this.locks.remove(polledOperation.getId().getValue());
                    }
                    return;
                }
            });
        }
    }

    private void resetLongPollingSession() {
        OperationSubscriber subscriber = (OperationSubscriber)this.ctx.getBean(OperationSubscriber.class);
        subscriber.resubscribe();
    }

    void executeOperation(OperationRepresentation operationRepresentation) {
        List handlers = this.operationHandlers.stream().filter(handler -> handler.supports(operationRepresentation)).collect(Collectors.toList());
        if (handlers.isEmpty()) {
            this.setOperationFailedWithReason(operationRepresentation.getId(), "Operation is not supported!");
            return;
        }
        for (OperationHandler handler2 : handlers) {
            BaseOperation operation = (BaseOperation)operationRepresentation.get(handler2.getSupportedOperationType());
            operation.setOperationId(operationRepresentation.getId());
            operation.setDeviceId(operationRepresentation.getDeviceId());
            handler2.handle(operation);
        }
    }

    private void setOperationFailedWithReason(GId operationId, String reason) {
        OperationRepresentation operation = new OperationRepresentation();
        operation.setId(operationId);
        operation.setStatus(OperationStatus.FAILED.name());
        operation.setFailureReason(reason);
        try {
            this.deviceControlApi.update(operation);
        }
        catch (SvensonRuntimeException e) {
            log.warn("Ignoring json parsing error on operation update - this is expected when operation JSON is invalid");
            log.warn(e.getMessage());
        }
    }

    public void afterPropertiesSet() {
        log.info("Registering operation handlers... ");
        this.operationHandlers.clear();
        Map allHandlers = this.ctx.getBeansOfType(OperationHandler.class);
        allHandlers.forEach((key, value) -> {
            log.info("Adding operation handler: {}", key);
            this.operationHandlers.add(value);
        });
        log.info("Initializing operation executor with queue size: {}", (Object)this.generalConfiguration.getOperationQueueSize());
        this.pendingOperations = new ArrayBlockingQueue(this.generalConfiguration.getOperationQueueSize(), true);
    }

    private boolean isOperationInTheQueue(OperationRepresentation operationRepresentation) {
        if (this.pendingOperations.contains(operationRepresentation)) {
            return true;
        }
        return this.pendingOperations.stream().anyMatch(operation -> operation.getId().equals((Object)operationRepresentation.getId()));
    }

    private boolean isOperationOutdated(OperationRepresentation operation) {
        DateTime createdAt = operation.getCreationDateTime();
        return !Objects.isNull(createdAt) && createdAt.plusMinutes(1).isBeforeNow();
    }

    public ExpiringMap<String, DateTime> getExecutedOperationsCache() {
        return this.executedOperationsCache;
    }
}

